import logging
import asyncio
import math

from eth_utils import to_text
from web3 import Web3
import aiohttp
from web3.utils.encoding import FriendlyJsonSerde

from gateway.utils import DictToHexBytes

logger = logging.getLogger('django-gateway.gateway')


def generate_http_gateway(url, is_poa):
    provider_generator = HTTPProviderGenerator(url)
    return Gateway(provider_generator, is_poa)


class HTTPProviderGenerator:

    def __init__(self, url):
        self.url = url

    def generate_provider(self):
        return Web3.HTTPProvider(self.url)


class InvalidTransactionException(Exception):
    pass


class NoConnectionException(Exception):
    pass


class Gateway:

    def __init__(self, provider_generator, is_geth_poa_network=False):
        self.w3 = Web3(provider_generator.generate_provider())
        if is_geth_poa_network:
            from web3.middleware import geth_poa_middleware
            self.w3.middleware_stack.inject(geth_poa_middleware, layer=0)

        if not self.w3.isConnected():
            raise InvalidTransactionException("No connection to node")

    def get_balance(self, address, measure='wei'):
        return self.w3.fromWei(self.w3.eth.getBalance(self.to_checksum_address(address)), measure)

    # precondición: la dirección es válida
    def get_contract(self, abi, address):
        return self.w3.eth.contract(abi=abi, address=address)

    # precondición: la dirección es válida
    def get_contract_functions(self, abi, address):
        contrato = self.get_contract(abi, address)
        return contrato.functions

    # precondición: la dirección es válida
    def get_functions(self, contract):
        return contract.functions

    # precondición: la dirección es válida
    def get_contract_events(self, abi, address):
        contrato = self.get_contract(abi=abi, address=address)
        return contrato.events

    def get_events(self, contract):
        return contract.events

    def is_valid_address(self, address):
        return self.w3.isAddress(address)

    def to_checksum_address(self, address):
        return self.w3.toChecksumAddress(address)

    def get_block(self, block_number):
        return self.w3.eth.getBlock(block_number)

    def get_tx_receipt(self, tx_hash):
        return self.w3.eth.getTransactionReceipt(tx_hash)

    def get_transaction(self, tx_hash):
        return self.w3.eth.getTransaction(tx_hash)

    def get_last_blocknumber(self):
        return self.w3.eth.blockNumber

    def get_node_accounts(self):
        return self.w3.eth.accounts

    def is_connected(self):
        return self.w3.eth.isConnected()

    def execute_transaction(self, function, transaction_data=None, exception_message=""):
        logger.debug("Executing transaction for function {} / tx data {}".format(str(function), str(transaction_data)))
        if transaction_data is None:
            transaction_data = {}
        try:
            function.call(transaction_data)
        except ValueError as ve:
            raise InvalidTransactionException(ve.args[0]['message'])
        tx_hash = function.transact(transaction_data)
        logger.debug("Transaction hash: {}".format(tx_hash.hex()))
        return tx_hash

    def call_transaction(self, function, transaction_data=None, exception_message=""):
        if transaction_data is None:
            transaction_data = {}
        res = function.call(transaction_data)
        return res

    def search_entries(self, event, params):
        logger.debug("Searching entries for event {} / params : {}".format(str(event), str(params)))
        if 'fromBlock' not in params:
            params['fromBlock'] = 0
        if 'toBlock' not in params:
            params['toBlock'] = 'latest'
        event_filter = event.createFilter(**params)
        return event_filter.get_all_entries()

    def get_undeployed_contract(self, abi, bytecode):
        return self.w3.eth.contract(abi=abi, bytecode=bytecode)

    def get_receipt(self, tx, wait=False):
        rcpt = self.w3.eth.getTransactionReceipt(tx)
        if wait:
            rcpt = self.w3.eth.waitForTransactionReceipt(tx)
        return rcpt

    def unlock_account(self, account, passphrase):
        logger.debug("Unlocking account for {}".format(str(account)))
        return self.w3.personal.unlockAccount(account, passphrase)

    def get_event_abi(self, event):
        return

    def get_tx_input(self, contract, tx):
        return contract.decode_function_input(tx.input)

    def transaction_is_canonical(self, transaction):
        last_block_number = self.get_last_blocknumber()
        tx_block_number = transaction.blockNumber
        required_block_difference = math.ceil(self.get_signers_count() / 2) + 1
        return (last_block_number - tx_block_number) > required_block_difference

    def pending_transactions(self):
        pending = self.w3.txpool.content['pending']
        txs = []
        for account, transactions in pending.items():
            for nonce, tx in transactions.items():
                txs.append(tx)
        return txs

    def get_signers_count(self):
        return 10

    def get_smart_contract_bytecode(self, smart_contract_address):
        return self.w3.eth.getCode(smart_contract_address)


class ContractInteractor:
    gateway = ...  # type: Gateway
    searcher = ...  # type: HTTPFastLogSearcher

    def __init__(self, gateway, abi, address, searcher=None):
        self.searcher = searcher
        self.abi = abi
        self.contract_address = address
        self.gateway = gateway

    def execute_function(self, function, tx_params, error_message=''):
        res = self.gateway.execute_transaction(function, tx_params, error_message)
        return res.hex()

    def search_logs(self, event_name, event_params):
        logger.debug("Searching logs for {} with params {}".format(str(event_name), str(event_params)))
        contract = self.gateway.get_contract(self.abi, self.contract_address)
        events = self.gateway.get_events(contract)
        event = getattr(events, event_name)
        if self.searcher is not None:
            entries = self.searcher.search_entries(event, event_params)
        else:
            entries = self.gateway.search_entries(event, event_params)
        return entries

    def get_function(self, function_name):
        return getattr(self.gateway.get_functions(self.gateway.get_contract(self.abi, self.contract_address)),
                       function_name)

    def call_function(self, function, tx_params={}, error_message=''):
        res = self.gateway.call_transaction(function, tx_params, error_message)
        return res

    def decode_function_input(self, tx):
        contract = self.gateway.get_contract(self.abi, self.contract_address)
        return self.gateway.get_tx_input(contract, tx)


class ContractDeployer:
    gateway = ...  # type: Gateway

    def __init__(self, gateway):
        self.gateway = gateway

    def deploy_contract(self, abi, bytecode, gas, from_account, wait=False):
        logger.debug("Deploying contract from account: {}".format(str(from_account)))
        contrato = self.gateway.get_undeployed_contract(abi, bytecode)

        # Submit the transaction that deploys the contract
        tx_hash = contrato.constructor().transact({'from': from_account, 'gas': gas, 'gasLimit': gas})
        logger.debug("Transaction hash for contract deploy : {}".format(str(tx_hash.hex())))
        # Wait for the transaction to be mined, and get the transaction receipt
        tx_receipt = self.gateway.get_receipt(tx_hash, wait)
        logger.debug("Receipt for contract deploy: {}".format(str(tx_receipt)))
        return tx_receipt

    def estimate_deploy(self, abi, bytecode):
        contrato = self.gateway.get_undeployed_contract(abi, bytecode)
        return contrato.constructor().estimateGas()


class HTTPFastLogSearcher:
    coroutines = 10
    gateway = ...  # type: Gateway
    url = ...  # type: str
    to_hex_dict = DictToHexBytes()

    def __init__(self, **kwargs):
        for k, v in kwargs.items():
            setattr(self, k, v)

    def encode_rpc_request(self, method, params, rpc_id):
        rpc_dict = {
            "jsonrpc": "2.0",
            "method": method,
            "params": params or [],
            "id": rpc_id,
        }
        return rpc_dict

    async def _async_search_entries(self, session, event, params, json_rpc_id, res_args):
        logger.debug("Starting coroutine id : {}".format(json_rpc_id))
        logger.debug("Creating filter with params : {}".format(str(params)))
        event_filter = event.createFilter(**params)
        logger.debug("Encoding json rpc request for filter id: {} / json_rpc_id : {}"
                     .format(str(event_filter.filter_id), str(json_rpc_id)))
        rpc_params = self.encode_rpc_request('eth_getFilterLogs', [event_filter.filter_id], json_rpc_id)
        logger.debug("Requesting POST to url: {}".format(str(self.url)))
        async with session.post(self.url, json=rpc_params) as raw_response:
            logger.debug("Awaiting for response of id : {}".format(str(json_rpc_id)))
            data = await raw_response.read()  # necesito los bytes, no uso el .json() a proposito
            text_response = to_text(data)  # hago esto porque asi puedo decodear usando los metodos de web3
            decoded_data = FriendlyJsonSerde().json_decode(text_response)
            logger.debug("Data assigned: {}".format(str(data)))
            if 'result' in decoded_data and len(decoded_data['result']) > 0:
                logger.debug("Decoding response of id : {}".format(str(json_rpc_id)))
                res_args[json_rpc_id] = []
                for r in decoded_data['result']:
                    logger.debug("Adding response for id : {}".format(str(json_rpc_id)))
                    self.to_hex_dict.hex_dict(r)
                    res_args[json_rpc_id].append(event_filter.format_entry(r))  # me desentiendo de la interpretacion
                    # con w3

    async def _async_search(self, event, params, amount, from_block, resto, res):
        logger.debug("Creating aiohttp session")
        headers = {'Content-Type': 'application/json', }
        async with aiohttp.ClientSession(headers=headers) as session:
            tasks = []
            logger.debug("Looping to {}".format(str(self.coroutines)))
            for x in range(self.coroutines):
                p = params.copy()
                p['fromBlock'] = int(amount * x) + from_block
                p['toBlock'] = int(amount * (x + 1)) + from_block
                tasks.append(self._async_search_entries(session, event, p, x, res))
            if resto != 0:
                p = params.copy()
                p['fromBlock'] = int(amount * self.coroutines) + from_block
                p['toBlock'] = 'latest'
                tasks.append(self._async_search_entries(session, event, p, self.coroutines + 1, res))
            logger.debug("Gathering tasks")
            await asyncio.gather(*tasks)

    def search_entries(self, event, params):
        logger.debug("Searching entries fast for event: {} / params : {}".format(str(event), str(params)))
        from_block = params['fromBlock']
        to_block = params['toBlock']
        if 'fromBlock' not in params:
            params['fromBlock'] = 0
            from_block = 0
        if 'toBlock' not in params:
            params['toBlock'] = 'latest'
        loop = asyncio.get_event_loop()
        logger.debug("Search from block: {}".format(str(from_block)))
        if params['toBlock'] == 'latest':
            to_block = self.gateway.get_last_blocknumber()
            logger.debug("Search until latest block : {}".format(str(to_block)))
        amount = int((to_block - from_block) / self.coroutines)
        resto = int((to_block - from_block) % self.coroutines)
        logger.debug("Block amount per coroutine : {}".format(str(amount)))
        if amount < self.coroutines:
            logger.debug("Amount is less than coroutines number")
            results = self.gateway.search_entries(event, params)
            return results
        else:
            results = {}
            logger.debug("Running loop until complete")
            loop.run_until_complete(self._async_search(event, params, amount, from_block, resto, results))
            res = []
            for k, v in results.items():
                for i in v:
                    res.append(i)
            return res
